import numpy as np
import time
import socket
import pickle
import threading
import random

# -------------------------------
# Multi-Core HDGL Lattice
# -------------------------------
NUM_CORES = 2
NUM_STRANDS = 8
SLOTS_PER_STRAND = 4
TOTAL_SLOTS = NUM_STRANDS * SLOTS_PER_STRAND

PHI = 1.6180339887
SQRT_PHI = np.sqrt(PHI)
N_BASE = np.arange(1, NUM_STRANDS+1)
OMEGA_BASE = 1 / (PHI**N_BASE)**7

cores = []
for c in range(NUM_CORES):
    lattice = np.random.uniform(0.5, 1.0, (NUM_STRANDS, SLOTS_PER_STRAND))
    phases = np.random.uniform(0, 2*np.pi, (NUM_STRANDS, SLOTS_PER_STRAND))
    weights = np.ones((NUM_STRANDS, SLOTS_PER_STRAND))
    cores.append({'lattice': lattice, 'phases': phases, 'weights': weights, 'omega': OMEGA_BASE*(1/(c+1))})

# -------------------------------
# Networking
# -------------------------------
UDP_PORT = 5005
BROADCAST_IP = "255.255.255.255"
sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
sock.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)
sock.bind(("", UDP_PORT))
sock.settimeout(0.001)

peers = {}  # addr -> last seen + timestamp + lattice snapshot

def broadcast_lattice(gps_time):
    data = pickle.dumps({'cores': cores, 'timestamp': gps_time})
    sock.sendto(data, (BROADCAST_IP, UDP_PORT))

def listen_for_peers():
    global cores
    while True:
        try:
            data, addr = sock.recvfrom(8192)
            packet = pickle.loads(data)
            peer_cores = packet['cores']
            peer_time = packet['timestamp']

            peers[addr[0]] = {'time': peer_time, 'cores': peer_cores}

        except socket.timeout:
            continue
        except Exception as e:
            print(f"Network error: {e}")

listener_thread = threading.Thread(target=listen_for_peers, daemon=True)
listener_thread.start()

# -------------------------------
# Adaptive Phase Correction
# -------------------------------
PHASE_CORRECTION_GAIN = 0.1  # How aggressively we adjust
def adaptive_phase_correction():
    global cores
    while True:
        for addr, peer in peers.items():
            peer_cores = peer['cores']
            peer_time = peer['time']
            delay = time.time() - peer_time

            for c, peer_core in enumerate(peer_cores):
                local_core = cores[c]
                # Calculate phase difference
                phase_diff = (peer_core['phases'] - local_core['phases']) % (2*np.pi)
                # Adjust local phase with small fraction of the difference
                local_core['phases'] += PHASE_CORRECTION_GAIN * phase_diff
                # Optional: weight update based on signal similarity
                local_core['weights'] = 0.9*local_core['weights'] + 0.1*peer_core['weights']
        time.sleep(0.05)  # run ~20Hz for real-time adjustment

correction_thread = threading.Thread(target=adaptive_phase_correction, daemon=True)
correction_thread.start()

# -------------------------------
# Frequency Hopping & RF Mapping
# -------------------------------
HOP_INTERVAL = 0.1
FREQ_HOP_RANGES = [(100e3, 200e3), (200e3, 300e3), (300e3, 400e3)]
FS = 1_000_000
BLOCK_SIZE = 4096
t = np.arange(BLOCK_SIZE) / FS

def get_hopped_frequencies():
    freqs = []
    for idx in range(TOTAL_SLOTS * NUM_CORES):
        hop_range = random.choice(FREQ_HOP_RANGES)
        freqs.append(random.uniform(*hop_range))
    return np.array(freqs)

def generate_rf_block(freqs):
    rf_block = np.zeros(BLOCK_SIZE, dtype=np.complex64)
    for c, core in enumerate(cores):
        lattice = core['lattice']
        phases = core['phases']
        weights = core['weights']
        omega = core['omega']
        lattice_new = np.copy(lattice)
        for s in range(NUM_STRANDS):
            for i in range(SLOTS_PER_STRAND):
                lattice_new[s,i] += 0.02*omega[s]
        core['lattice'][:] = lattice_new
        weights[:] = 0.9*weights + 0.1*lattice_new
        phases[:] += 0.05*lattice
        for idx in range(TOTAL_SLOTS):
            strand = idx // SLOTS_PER_STRAND
            slot = idx % SLOTS_PER_STRAND
            amp = lattice[strand, slot] / np.max(lattice)
            phi = phases[strand, slot]
            freq_offset = 50e3 * (lattice[strand, slot] - 0.5)
            carrier = np.exp(1j*(2*np.pi*(freqs[c*TOTAL_SLOTS + idx]+freq_offset)*t + phi))
            rf_block += amp * carrier
    rf_block /= np.max(np.abs(rf_block))
    return rf_block

# -------------------------------
# Peer Cleanup
# -------------------------------
def prune_peers(timeout=5):
    while True:
        now = time.time()
        stale = [addr for addr,last_seen in peers.items() if now-last_seen['time']>timeout]
        for addr in stale:
            del peers[addr]
        time.sleep(1)

prune_thread = threading.Thread(target=prune_peers, daemon=True)
prune_thread.start()

# -------------------------------
# Main Geo-Sync Loop
# -------------------------------
try:
    print("Adaptive Phase-Corrected Geo-Synchronized HDGL RF Streaming.")
    last_hop = time.time()
    freqs = get_hopped_frequencies()

    while True:
        gps_time = time.time()  # replace with GPS/atomic clock
        if time.time() - last_hop > HOP_INTERVAL:
            freqs = get_hopped_frequencies()
            last_hop = time.time()
        block = generate_rf_block(freqs)
        broadcast_lattice(gps_time)
        # sdr.write_samples(block)  # connect to SDR hardware
        time.sleep(BLOCK_SIZE/FS)

except KeyboardInterrupt:
    print("Streaming stopped.")
